<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Notification.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Notification
 * @description     : Notification send users setting.  
 * @author          : South Bangla IT Zone Team
 * @url             : https://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Notification extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Setting_Model', 'setting', true);
    }


    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load notification setting user interface
     *                    and send user notification processing
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        if ($this->input->server('REQUEST_METHOD') == "POST") {
            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim');

            $notifications = $this->input->post('notification[]');

            foreach ($notifications as $value) {
                $is_email = 0;
                $is_sms = 0;
                $email = $this->input->post('is_email_' . $value);

                if (isset($email)) {
                    $is_email = 1;
                }
                $sms = $this->input->post('is_sms_' . $value);
                if (isset($sms)) {
                    $is_sms = 1;
                }

                $data = array(
                    'is_email' => $is_email,
                    'is_sms' => $is_sms
                );
                $this->setting->update('notifications', $data, array('id' => $value));
            }
        }

        $this->data['notifications'] = $this->setting->get_list('notifications', array('status' => 1), '', '', '', 'id', 'ASC');
        $this->layout->title($this->lang->line('notification') . ' ' . $this->lang->line('setting') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('notification/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add Notification" user interface
     *                    and store "notifications" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_notification_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_notification_data();

                $insert_id = $this->setting->insert('notifications', $data);
                if ($insert_id) {
                    create_log('Has been created a notification : ' . $data['type']);
                    success($this->lang->line('insert_success'));
                    redirect('setting/notification');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('setting/notification/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('notification') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('notification/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "notification" user interface
     *                    with populated "notifications" value
     *                    and update "notifications" database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_notification_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_notification_data();
                $updated = $this->setting->update('notifications', $data, array('id' => $this->input->post('id')));

                if ($updated) {

                    create_log('Has been updated a notification : ' . $data['type']);

                    success($this->lang->line('update_success'));
                    redirect('setting/notification');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('setting/notification/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['notification'] = $this->setting->get_single('notifications', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['notification'] = $this->setting->get_single('notifications', array('id' => $id));

                if (!$this->data['notification']) {
                    redirect('setting/notification');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('notification') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('notification/edit', $this->data);
    }

    /*****************Function _prepare_notification_validation**********************************
     * @type            : Function
     * @function name   : _prepare_notification_validation
     * @description     : Process "notification" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_notification_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');

        $this->form_validation->set_rules('type', $this->lang->line('notification') . ' ' . $this->lang->line('type'), 'trim|required|callback_type');
        $this->form_validation->set_rules('message', $this->lang->line('message'), 'trim');
    }

    /*****************Function type**********************************
     * @type            : Function
     * @function name   : type
     * @description     : Unique check for "Notification Type" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function type() {
        if ($this->input->post('id') == '') {
            $notification = $this->setting->duplicate_check_notification_type($this->input->post('type'));
            if ($notification) {
                $this->form_validation->set_message('type', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $notification = $this->setting->duplicate_check_notification_type($this->input->post('type'), $this->input->post('id'));
            if ($notification) {
                $this->form_validation->set_message('type', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_notification_data**********************************
     * @type            : Function
     * @function name   : _get_posted_notification_data
     * @description     : Prepare "Notification" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_notification_data() {

        $items = array();
        $items[] = 'type';
        $items[] = 'message';
        $data = elements($items, $_POST);

        $data['type'] = $this->input->post('type') ? str_replace(array(' ', '-', '--'), '_', $this->input->post('type')) : 0;
        $data['is_email'] = $this->input->post('is_email') ? $this->input->post('is_email') : 0;
        $data['is_sms'] = $this->input->post('is_sms') ? $this->input->post('is_sms') : 0;

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Notification" from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);
        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('setting/notification');
        }

        $notification = $this->setting->get_single('notifications', array('id' => $id));

        if ($this->setting->delete('notifications', array('id' => $id))) {
            create_log('Has been deleted a notification : ' . $notification->type);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('setting/notification');
    }
}
