<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************News.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : News
 * @description     : Manage news.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class News extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('News_Model', 'news', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "News List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        if ($_GET && $this->input->get('date')) {
            $keyword = trim($this->input->get('keyword'));
            $date = $this->input->get('date');
            $exp = explode('-', $date);
            $form_date = date('Y-m-d', strtotime($exp[0]));
            $to_date = date('Y-m-d', strtotime($exp[1]));
            $created_by = $this->input->get('created_by');

            $config = array();
            $config = get_pagination(ADMIN_PER_PAGE);
            $config['base_url'] = site_url('announcement/news/?keyword=' . $keyword . '&date=' . $date . '&created_by=' . $created_by);
            if ($this->input->get('page')) {
                $sgm = (int) trim($this->input->get('page'));
                $offset = $config['per_page'] * ($sgm - 1);
            } else {
                $offset = 0;
            }
            $config['total_rows'] = $this->news->get_news_list($config['per_page'], $offset, $keyword, $form_date, $to_date, $created_by, $count = TRUE);
            $this->pagination->initialize($config);
            $this->data['news'] = $this->news->get_news_list($config['per_page'], $offset, $keyword, $form_date, $to_date, $created_by, $count = FALSE);
            $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));

            $this->data['keyword'] = $keyword;
            $this->data['date'] = $date;
            $this->data['created_by'] = $created_by;
        } else {
            $config = array();
            $config = get_pagination(ADMIN_PER_PAGE);
            $config['base_url'] = current_url();
            if ($this->input->get('page')) {
                $sgm = (int) trim($this->input->get('page'));
                $offset = $config['per_page'] * ($sgm - 1);
            } else {
                $offset = 0;
            }

            $config['total_rows'] = $this->news->get_news_list($config['per_page'], $offset, null, null, null, null, $count = TRUE);
            $this->pagination->initialize($config);
            $this->data['news'] = $this->news->get_news_list($config['per_page'], $offset, null, null, null, null, $count = FALSE);
            $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        }

        $this->data['employees'] = $this->news->get_employee_list();
        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_news') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('news/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new News" user interface
     *                    and process to store "news" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_news_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_news_data();
                $insert_id = $this->news->insert('news', $data);
                if ($insert_id) {
                    create_log('Has been created a News : ' . $data['title']);
                    success($this->lang->line('insert_success'));
                    redirect('announcement/news');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('announcement/news/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('news') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('news/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "News" user interface
     *                    with populate "News" value
     *                    and process to update "news" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_news_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_news_data();
                $updated = $this->news->update('news', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a News : ' . $data['title']);
                    success($this->lang->line('update_success'));
                    redirect('announcement/news');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('announcement/news/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['news'] = $this->news->get_single('news', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['news'] = $this->news->get_single('news', array('id' => $id));
                if (!$this->data['news']) {
                    redirect('announcement/news');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('news') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('news/edit', $this->data);
    }


    /*     * ***************Function get_single_news**********************************
     * @type            : Function
     * @function name   : get_single_news
     * @description     : "Load single news information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_news() {
        $id = $this->input->post('id');
        $this->data['news'] = $this->news->get_single_news($id);
        echo $this->load->view('news/view', $this->data);
    }

    /*****************Function _prepare_news_validation**********************************
     * @type            : Function
     * @function name   : _prepare_news_validation
     * @description     : Process "News" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_news_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('title', $this->lang->line('title'), 'trim|required');
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required');
    }

    /*****************Function _get_posted_news_data**********************************
     * @type            : Function
     * @function name   : _get_posted_news_data
     * @description     : Prepare "News" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_news_data() {

        $items = array();
        $items[] = 'title';
        $items[] = 'description';
        $items[] = 'is_view_on_web';
        $items[] = 'status';
        $data = elements($items, $_POST);

        $data['date'] = $this->input->post('date') ? date('Y-m-d', strtotime($this->input->post('date'))) : date('Y-m-d');

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if ($_FILES['attachment']['name']) {
            $data['attachment'] = $this->_upload_attachment();
        }

        return $data;
    }

    /*     * ***************Function _upload_attachment**********************************
     * @type            : Function
     * @function name   : _upload_attachment
     * @description     : Process to upload news attachment into server
     *                     and return attachment file name
     * @param           : null
     * @return          : $return_attachment string value
     * ********************************************************** */

    private function _upload_attachment() {

        $prev_attachment = $this->input->post('prev_attachment');
        $attachment = $_FILES['attachment']['name'];
        $attachment_type = $_FILES['attachment']['type'];
        $return_attachment = '';

        if ($attachment != "") {
            if (
                $attachment_type == 'application/vnd.openxmlformats-officedocument.wordprocessingml.document' ||
                $attachment_type == 'application/msword' || $attachment_type == 'text/plain' ||
                $attachment_type == 'application/vnd.ms-office' || $attachment_type == 'application/pdf' ||
                $attachment_type == 'image/jpeg' || $attachment_type == 'image/pjpeg' ||
                $attachment_type == 'image/jpg' || $attachment_type == 'image/png' ||
                $attachment_type == 'image/x-png' || $attachment_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/news/';

                $file_type = explode(".", $attachment);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $attachment_path = 'news-attachment-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['attachment']['tmp_name'], $destination . $attachment_path);

                // need to unlink previous attachment
                if ($prev_attachment != "") {
                    if (file_exists($destination . $prev_attachment)) {
                        @unlink($destination . $prev_attachment);
                    }
                }

                $return_attachment = $attachment_path;
            }
        } else {
            $return_attachment = $prev_attachment;
        }

        return $return_attachment;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "News" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('announcement/news');
        }

        $news = $this->news->get_single('news', array('id' => $id));

        if ($this->news->delete('news', array('id' => $id))) {
            // delete attachment
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/news/' . $news->attachment)) {
                @unlink($destination . '/news/' . $news->attachment);
            }
            create_log('Has been deleted a News : ' . $news->title);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
