<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Shift.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Shift
 * @description     : Manage shift.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Shift extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Shift_Model', 'shift', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Shift List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $keyword = $this->input->get('keyword');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($keyword) {
            $config['base_url'] = site_url('hrm/shift/?keyword=' . $keyword);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->shift->get_shift_list($config['per_page'], $offset, $keyword, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['shifts'] = $this->shift->get_shift_list($config['per_page'], $offset, $keyword, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['keyword'] = $keyword;


        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_shift') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('shift/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Shift" user interface
     *                    and process to store "shifts" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_shift_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_shift_data();
                $insert_id = $this->shift->insert('shifts', $data);
                if ($insert_id) {
                    create_log('Has been created a shift : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('hrm/shift');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hrm/shift/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->global_setting->brand_title);
        $this->layout->view('shift/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Shift" user interface
     *                    with populate "Shift" value
     *                    and process to update "shifts" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_shift_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_shift_data();
                $updated = $this->shift->update('shifts', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a shift : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('hrm/shift');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hrm/shift/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['shift'] = $this->shift->get_single('shifts', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['shift'] = $this->shift->get_single('shifts', array('id' => $id));
                if (!$this->data['shift']) {
                    redirect('hrm/shift');
                }
            }
        }
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->global_setting->brand_title);
        $this->layout->view('shift/edit', $this->data);
    }

    /*     * ***************Function get_single_shift**********************************
     * @type            : Function
     * @function name   : get_single_shift
     * @description     : "Load single shift information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_shift() {
        $id = $this->input->post('id');
        $this->data['shift'] = $this->shift->get_single_shift($id);
        echo $this->load->view('hrm/shift/view', $this->data);
    }

    /*****************Function _prepare_shift_validation**********************************
     * @type            : Function
     * @function name   : _prepare_shift_validation
     * @description     : Process "Shift" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_shift_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('name', $this->lang->line('shift') . ' ' . $this->lang->line('name'), 'trim|required|callback_name');
        $this->form_validation->set_rules('in_time', $this->lang->line('in_time'), 'trim|required');
        $this->form_validation->set_rules('out_time', $this->lang->line('out_time'), 'trim|required');
        $this->form_validation->set_rules('absent_time', $this->lang->line('absent_time'), 'trim|required');
        $this->form_validation->set_rules('absent_time', $this->lang->line('absent_time'), 'trim|required');
    }

    /*****************Function name**********************************
     * @type            : Function
     * @function name   : name
     * @description     : Unique check for "Shift Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function name() {
        if ($this->input->post('id') == '') {
            $shift = $this->shift->duplicate_check($this->input->post('name'));
            if ($shift) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $shift = $this->shift->duplicate_check($this->input->post('name'), $this->input->post('id'));
            if ($shift) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_shift_data**********************************
     * @type            : Function
     * @function name   : _get_posted_shift_data
     * @description     : Prepare "Shift" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_shift_data() {

        $items = array();
        $items[] = 'name';
        $items[] = 'bn_name';
        $items[] = 'weekend';
        $items[] = 'remark';
        $data = elements($items, $_POST);

        $data['in_time'] = $this->input->post('in_time') ? date('H:i:s', strtotime($this->input->post('in_time'))) : '09:00:00';
        $data['out_time'] = $this->input->post('out_time') ? date('H:i:s', strtotime($this->input->post('out_time'))) : '16:00:00';
        $data['absent_time'] = $this->input->post('absent_time') ? date('H:i:s', strtotime($this->input->post('absent_time'))) : '16:00:00';
        $data['lunch_out_time'] = $this->input->post('lunch_out_time') ? date('H:i:s', strtotime($this->input->post('lunch_out_time'))) : '16:00:00';
        $data['lunch_in_time'] = $this->input->post('lunch_in_time') ? date('H:i:s', strtotime($this->input->post('lunch_in_time'))) : '16:00:00';
        $data['late_time'] = $this->input->post('late_time') ? date('H:i:s', strtotime($this->input->post('late_time'))) : '16:00:00';

        $data['late_allow'] = $this->input->post('late_allow') ?? 0;
        $data['special_late_allow'] = $this->input->post('special_late_allow') ?? 0;
        $data['early_out_allow'] = $this->input->post('early_out_allow') ?? 0;
        $data['out_next_day'] = $this->input->post('out_next_day') ?? 0;
        $data['status'] = $this->input->post('status') ?? 0;

        $in_time = new DateTime($data['in_time']);
        $out_time = new DateTime($data['out_time']);
        $diff = $in_time->diff($out_time);
        $data['work_hour'] =  $diff->format("%H:%I:%S");

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }
        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Shift" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hrm/shift');
        }

        $shift = $this->shift->get_single('shifts', array('id' => $id));
        if (isset($shift) && $id > 0) {
            $this->db->trans_start();
            create_log('Has been deleted a shift : ' . $shift->name);
            $this->shift->delete('shifts', array('id' => $id));
            $this->db->trans_complete();
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
